<?php

require_once dirname(__FILE__).'/ILoggable.php';
require_once dirname(__FILE__).'/SiteproApiClient.php';

class SiteproBuilder extends SiteProApiClient implements ILoggable {
	
	/** @var CPANEL */
	private $cpanel;
	/** @var CpanelApi */
	private $cpanelApi;
	/** @var array */
	private static $translations;
	/** @var boolean */
	private $debug = false;
	/** @var boolean */
	private $fromWhmcs = false;
	/** @var string */
	private $lang;
	/** @var string|null */
	private $serverAddr;

	public function __construct($apiUrl, $apiUsername, $apiPassword) {
		parent::__construct($apiUrl, $apiUsername, $apiPassword);
		$this->debug = (isset($_GET['debug']) && $_GET['debug'] == 'true');
		$this->fromWhmcs = (defined('SITEPROBUILDER_WHMCS') && SITEPROBUILDER_WHMCS);
		if ($this->fromWhmcs) {
			require_once dirname(__FILE__).'/CpanelApi.php';
			$this->cpanelApi = new CpanelApi();
			$this->cpanelApi->setLogging($this);
		} else {
			require_once "/usr/local/cpanel/php/cpanel.php";
			$this->cpanel = new CPANEL();
		}
		if ($this->isLogEnabled()) {
			error_reporting(E_ALL);
			ini_set('display_errors', true);
		}
		$this->log($_SERVER, 'SERVER');
		$this->log($apiUrl, 'API URL');
	}
	
	public function log($msg, $title = null) {
		if ($this->isLogEnabled()) {
			if ($title) echo '<strong>'.$title.':</strong><br />';
			echo '<pre>'.(is_string($msg) ? $msg : print_r($msg, true)).'</pre><br />';
		}
	}
	
	public function isLogEnabled() {
		return $this->debug;
	}
	
	public function langValue($key) {
		if (!self::$translations) {
			self::$translations = array();
			$lang = $this->getLang();
			if (!$lang) $lang = 'en';
			$locale_dir = dirname(__FILE__).'/locale';
			$locale_file = $locale_dir.'/'.$lang;
			if (!is_file($locale_file)) { $locale_file = $locale_dir.'/en'; }
			$locale_data = explode("\n", trim(file_get_contents($locale_file)));
			foreach ($locale_data as $li) {
				$tr = explode('=', $li, 2);
				if (!trim($tr[0]) || !isset($tr[1])) { continue; }
				self::$translations[trim($tr[0])] = trim($tr[1]);
			}
		}
		return isset(self::$translations[$key]) ? self::$translations[$key] : $key;
	}
	
	public function cpanelApi() {
		return $this->fromWhmcs ? $this->cpanelApi : $this->cpanel;
	}
	
	public function identifyWHMUser($baseUrl, $username, $password) {
		if ($this->fromWhmcs && $this->cpanelApi) {
			$this->cpanelApi->configure($baseUrl, $username, $password);
		}
	}
	
	public function identifyCpanelUser($username) {
		if ($this->fromWhmcs && $this->cpanelApi) {
			$this->cpanelApi->setCpanelUser($username);
		}
	}
	
	public function setServerAddr($serverAddr) {
		if ($this->fromWhmcs) {
			$this->serverAddr = $serverAddr;
		}
	}
	
	public function uapiCall($module, $function, $params = array()) {
		if ($this->fromWhmcs) {
			return $this->cpanelApi->call($module, $function, $params);
		} else {
			try {
				$resp = $this->cpanel->uapi($module, $function, $params);
				$res = $resp['cpanelresult']['result'];
				if (isset($res['status']) && $res['status'] != 1) {
					$error = 'API Error (2):';
					if (isset($resp['errors']) && $resp['errors']) {
						if (is_array($res['errors'])) {
							$error .= '<br />'.implode('<br />', $res['errors']);
						} else {
							$error = ' '.$res['errors'];
						}
					} else {
						$error .= ' unsuccessful call';
					}
					throw new ErrorException($error);
				}
			} catch (ErrorException $ex) {
				if (!$this->isLogEnabled()) {
					throw $ex;
				} else {
					$this->log($ex->getMessage(), 'API Error');
					exit();
				}
			}
			return isset($res['data']) ? $res['data'] : null;
		}
	}
	
	/**
	 * Generate strong password
	 * @param int $length password length
	 * @return string
	 */
	private function generatePassword($length = 9) {
		$sets = array('abcdefghijklmnopqrstuvwxyz', 'ABCDEFGHIJKLMNOPQRSTUVWXYZ', '0123456789', './');
		$password = '';
		foreach ($sets as $set) {
			$password .= $set[array_rand(str_split($set))];
		}
		$all = str_split(implode('', $sets));
		for ($i = 0, $c = ($length - count($sets)); $i < $c; $i++) { $password .= $all[array_rand($all)]; }

		return str_shuffle($password);
	}
	
	/**
	 * Load data from local storage
	 * @param string $path absolute path to local storage file
	 * @return stdClass data object
	 */
	private function localStorageLoad($path) {
		$data = new stdClass();
		if ($this->fromWhmcs) {
			try {
				$res = $this->cpanelApi->call('Fileman', 'get_file_content',
						array('dir' => dirname($path), 'file' => basename($path)));
				$dataStr = isset($res['content']) ? $res['content'] : null;
			} catch (ErrorException $ex) {}
		} else {
			if (is_file($path)) {
				$dataStr = file_get_contents($path);
			}
		}
		if (isset($dataStr)) {
			$dataObj = $dataStr ? json_decode($dataStr) : null;
			if (is_object($dataObj)) {
				$data = $dataObj;
			}
		}
		return $data;
	}
	
	/**
	 * Store data to local storage
	 * @param string $path absolute path to local storage file
	 * @param stdClass $data data object
	 * @return boolean true if success, false otherwise
	 */
	private function localStorageStore($path, $data) {
		$tmp = new stdClass();
		$dataStr = json_encode(is_object($data) ? $data : $tmp);
		if ($this->fromWhmcs) {
			try {
				$this->cpanelApi->call('Fileman', 'save_file_content',
						array('dir' => dirname($path), 'file' => basename($path), 'content' => $dataStr));
			} catch (ErrorException $ex) {}
		} else {
			if (file_put_contents($path, $dataStr) !== false) {
				chmod($path, 0600);
				return true;
			}
		}
		return false;
	}
	
	/**
	 * Check if FTP account exists
	 * @param stdClass $ftpData ftp data
	 * @return boolean
	 * @throws ErrorException
	 */
	private function ftpExists($ftpData) {
		if (!$ftpData || !is_object($ftpData) || !isset($ftpData->username) || !$ftpData->username) return false;
		$data = $this->uapiCall('Ftp', 'list_ftp');
		if ($data) {
			list($username) = explode('@', $ftpData->username);
			if (!strlen($username)) $username = $ftpData->username;
			foreach ($data as $li) {
				if ($li['type'] == 'logaccess') continue;
				if ($li['user'] == $username || $li['user'] == $ftpData->username) {
					return true;
				}
			}
		}
		return false;
	}
	
	/**
	 * Check if FTP account is working
	 * @param type $ftpData
	 * @param bool $useSSL
	 * @param bool $useGlobalHost
	 */
	private function ftpActive($ftpData, $useSSL = false, $useGlobalHost = false) {
		if (!$ftpData || !is_object($ftpData) || !isset($ftpData->username) || !$ftpData->username
				|| !isset($ftpData->password) || !$ftpData->password) return false;
		$host = ($useGlobalHost ? ($this->serverAddr ? $this->serverAddr : (isset($_SERVER['SERVER_ADDR']) && $_SERVER['SERVER_ADDR'])
				? $_SERVER['SERVER_ADDR'] : (isset($_SERVER['HTTP_HOST']) && $_SERVER['HTTP_HOST'])
					? $_SERVER['HTTP_HOST'] : null) : '127.0.0.1');
		if (!$host) return false;
		ob_start();
		$linkId = ($useSSL && function_exists('ftp_ssl_connect')) ? ftp_ssl_connect($host) : ftp_connect($host);
		if (!$linkId) {
			if (!$useGlobalHost) {
				return $this->ftpActive($ftpData, $useSSL, true);
			} else {
				return false;
			}
		}
		if (!ftp_login($linkId, $ftpData->username, $ftpData->password)) {
			if (!$useSSL) {
				if (preg_match('#cleartext\ sessions|tls|ssl#i', ob_get_clean())) {
					return $this->ftpActive($ftpData, true, $useGlobalHost);
				}
			} else {
				ob_end_clean();
				return false;
			}
		} else {
			ob_end_clean();
			return true;
		}
		return false;
	}
	
	/**
	 * Create new FTP account
	 * @param string $username username for ftp account
	 * @param string $domain domain to create FTP account for
	 * @return stdClass ftp data
	 * @throws ErrorException
	 */
	private function ftpCreate($username, $domain = null) {
		$ftpData = (object) array('username' => $username.($domain ? '@'.$domain : ''), 'password' => $this->generatePassword(16));
		$this->uapiCall('Ftp', 'add_ftp', array('user' => $ftpData->username, 'pass' => $ftpData->password, 'homedir' => '/'));
		return $ftpData;
	}
	
	private function getLang() {
		if (!$this->lang) {
			if ($this->fromWhmcs) {
				$lang2 = (isset($GLOBALS['_LANG']['locale']) ? $GLOBALS['_LANG']['locale'] : null);
				list($this->lang) = ($lang2 ? explode('_', $lang2, 2) : array(null));
			}
			if (!$this->lang) {
				try {
					$res = $this->uapiCall('Locale', 'get_attributes');
					$this->lang = $res['locale'];
				} catch (ErrorException $ex) {}
			}
		}
		return $this->lang;
	}
	
	public function openBuilder($domain, $user, $pass, $outputUrl = false) {
		$hostingPlan = null;
		$uploadDir = null;
		$domAddr = null;
		
		$res = $this->uapiCall('DomainInfo', 'single_domain_data', array('domain' => $domain));
		$uploadDir = ltrim(str_replace($res['homedir'], '', $res['documentroot']), '/');
		$lang = $this->getLang();
		if (!$lang) $lang = null;
		
		$apiUrl = (isset($_SERVER['SERVER_ADDR']) && $_SERVER['SERVER_ADDR']) ? $_SERVER['SERVER_ADDR'] : null;
		$res = $this->uapiCall('StatsBar', 'get_stats', array('display' => 'dedicatedip|sharedip|hostname|hostingpackage'));
		foreach ($res as $inf) {
			if (!$domAddr && ($inf['id'] == 'dedicatedip' || $inf['id'] == 'sharedip' || $inf['id'] == 'hostname')) {
				$domAddr = $inf['value'] ? $inf['value'] : null;
			} else if (!$hostingPlan && $inf['id'] == 'hostingpackage') {
				$hostingPlan = $inf['value'] ? $inf['value'] : null;
			}
		}
		if (!$apiUrl) $apiUrl = $domAddr;
		$localStorage = ((isset($_SERVER['HOME']) && $_SERVER['HOME']) ? $_SERVER['HOME'] : '/home/'.$user).'/.spbldr_localStorage';
		$this->log($localStorage, "Local storage path");
		if (!$pass || $pass == '__HIDDEN__') {
			$ls = $this->localStorageLoad($localStorage);
			$isset = (isset($ls->ftp));
			$exists = $isset && $this->ftpExists($ls->ftp);
			$active = $exists && $this->ftpActive($ls->ftp, false, $this->fromWhmcs);
			$this->log(array('isset' => $isset, 'exists' => $exists, 'active' => $active), "Retrieved FTP account information");
			try {
				if (!$isset || !$exists || !$active) {
					$ls->ftp = $this->ftpCreate(sprintf('%08x', crc32('spbldr...'.microtime())), $domain);
					$this->log($ls->ftp, "Created FTP account");
					$this->localStorageStore($localStorage, $ls);
				}
				if (isset($ls->ftp) && is_object($ls->ftp)) {
					$user = $ls->ftp->username;
					$pass = $ls->ftp->password;
					if (isset($ls->ftp->docRoot) && $ls->ftp->docRoot) $uploadDir = $ls->ftp->docRoot;
				}
			} catch (ErrorException $ex) {
				// handle errors if needed
				// echo 'Error: '.$ex->getMessage();
				// exit();
			}
		} else {
			$ls = $this->localStorageLoad($localStorage);
			try {
				if (!isset($ls->ftp) || $ls->ftp->username != $user || $ls->ftp->password != $pass) {
					$ls->ftp = (object) array('username' => $user, 'password' => $pass, 'native' => true);
					$this->localStorageStore($localStorage, $ls);
				}
			} catch (ErrorException $ex) {
				
			}
		}
		$params = array(
			"type" => "external",
			"domain" => $domain,
			"lang" => ($outputUrl ? null : $lang),
			"username" => $user,
			"password" => $pass,
			"apiUrl" => $domAddr,
			"hostingPlan" => $hostingPlan,
			"uploadDir" => $uploadDir,
			"panel" => ($this->fromWhmcs ? "WHMCS" : "WHM_cPanel"),
			"userId" => 587002
		);
		if ($this->serverAddr) {
			$params['serverIp'] = $this->serverAddr;
		}
		
		$usr = $this->remoteCall('requestLogin', $params);
		$this->log($usr, "Builder API response");
		if ($this->isLogEnabled()) exit();
		
		if (is_object($usr) && isset($usr->url) && $usr->url) {
			if ($outputUrl) {
				echo $usr->url;
			} else {
				header('Location: '.$usr->url, true);
			}
		} else if (is_object($usr) && isset($usr->error) && $usr->error) {
			echo 'Error: '.$usr->error->message;
		} else {
			echo 'Error: server error';
		}
		exit();
	}
	
	/** WHM has no curl installed by default */
	public function remoteCall($method, $params) {
		$useCurl = function_exists('curl_init');
		$this->log(($useCurl ? 'true' : 'false'), 'Use cURL');
		if ($useCurl) return parent::remoteCall($method, $params);
		
		if ($this->debug) {
			$url_parts = parse_url($this->apiUrl); $ip = null;
			if (isset($url_parts['host']) && $url_parts['host']) {
				$ip = gethostbyname($url_parts['host']);
			}
			$this->log($ip, "Builder API host IP");
		}
		$this->log($this->apiUrl.$method, "Builder API URL");
		
		$uinf = parse_url($this->apiUrl.$method);
		$request_host = $uinf['host'];
		$request_uri = $uinf['path'];
		$request_uri .= (isset($uinf['query']) && $uinf['query']) ? ('?'.$uinf['query']) : '';
		
		$errno = $errstr = null;
		$fp = fsockopen($request_host, (isset($uinf['port']) ? $uinf['port'] : 80), $errno, $errstr, 30);
		if (!$fp) {
			if ($errno === 0 && (gethostbyname($request_host) == $request_host)) {
				throw new ErrorException("Error: domain '$request_host' is not resolved.");
			}
			throw new ErrorException("$errstr ($errno)");
		} else {
			$post = json_encode($params);
			$content_length = mb_strlen($post, '8bit');
			
			$headers = "POST $request_uri HTTP/1.1\r\n";
			$headers .= "Host: $request_host\r\n";
			$headers .= "Connection: Close\r\n";
			$headers .= "Accept: text/html,application/json\r\n";
			$headers .= "User-Agent: Site.pro Builder plugin\r\n";
			$headers .= "Authorization: Basic ".base64_encode($this->apiUser.':'.$this->apiPass)."\r\n";
			$headers .= "Content-Type: application/json\r\n";
			$headers .= "Content-Length: $content_length\r\n";
			$headers .= "\r\n";
			fwrite($fp, $headers);
			if ($post) fwrite($fp, $post);
			$response_ = '';
			while (!feof($fp)) {
				$response_ .= fgets($fp);
			}
			fclose($fp);
			
			$response = explode("\r\n\r\n", $response_, 2);
			$result = array();
			$result['header']	= $response[0];
			$result['body']		= isset($response[1]) ? $response[1] : '';
			
			if (preg_match('#(?:\r\nTransfer-Encoding: chunked\r\n)#ism', $result['header'])) {
				$body = $result['body'];
				$result['body'] = '';
				$m = null;
				while (preg_match('#(?:\r\n|^)[0-9a-f]+\r\n#ism', $body, $m, PREG_OFFSET_CAPTURE)) {
					$size = intval(trim($m[0][0]), 16);
					if ($size) {
						$result['body'] .= substr($body, $m[0][1] + strlen($m[0][0]), $size);
					}
					$body = substr($body, $m[0][1] + strlen($m[0][0]) + $size);
				}
			}
			
			$http_code = preg_match('#^HTTP/1\.[0-9]+\ ([0-9]+)\ [^\ ]+.*#i', $result['header'], $m) ? $m[1] : 404;
			if ($http_code != 200) {
				$res = json_decode($result['body']);
				if (!$res) {
					$res = null;
					throw new ErrorException('Response Code ('.$http_code.')');
				}
			} else {
				$res = json_decode($result['body']);
			}
			return $res;
		}
		return null;
	}
	
}
