<?php


namespace api\helper;

use \MGModule\DomainAndEmailForwarding as main;
use MGModule\DomainAndEmailForwarding\models\Log;
/**
 * Description of ApiHelper
 *
 * @author Artur Pilch artur.pi@modulesgarden.com
 */
class ApiHelper
{

    protected $domain;
    protected $submodule;
    protected $domainId;
    protected $userId;

    public function __construct()
    {
        \MGModule\DomainAndEmailForwarding\mgLibs\MySQL\Query::useCurrentConnection();
        main\mgLibs\process\MainInstance::setInstanceName('MGModule\DomainAndEmailForwarding\Addon');
    }

    protected function domain()
    {
        if (!empty($this->domain))
        {
            return $this->domain;
        }
        $domain = new main\models\whmcs\Domain($this->domainId, [], $this->userId);
        return $this->domain = $domain;
    }

    protected function submodule()
    {
        if (!empty($this->submodule))
        {
            return $this->submodule;
        }
        $manager = new main\service\IntegrationManager();
        return $this->submodule = $manager->getActiveModule($this->domain());
    }

    public function getActiveModule(main\models\whmcs\Domain $domain)
    {
        try
        {
            $repository = new main\repository\SubmoduleConfigurationRepository();
            $conf       = $repository->findActive($domain->domainPricing()->domainSetting()->getSubmodule())
                ->fetchOne();
            return $this->getModule($conf->getClassName());
        }
        catch (\Exception $e)
        {
            return false;
        }
    }

    public function findActive($id)
    {
        $this->_filters['active'] = '1';
        $this->_filters['id']     = PdoWrapper::real_escape_string($id);
        return $this;
    }

    private function domainPolicy()
    {
        if (!$this->domain()->domainPricing()->domainSetting()->isEmailForwarding())
        {
            throw new main\mgLibs\exceptions\System("No access to 'Email Forwards' for that domain.");
        }

    }

    function userServices($id)
    {
        try
        {
            $search = [
                'id',
                'domain',
                'status',
                'expirydate',
                'nextduedate',
                'nextinvoicedate'
            ];
            return \MGModule\DomainAndEmailForwarding\mgLibs\MySQL\Query::select($search, 'tbldomains', ['userid' => $id])
                ->fetchAll();

        }
        catch (\Exception $ex)
        {
            Log::error(sprintf("API: Services list failed '%s' for User: %s - IP: %s\r\n",
                $ex->getMessage(),
                $id,
                $this->getRealIpAddr()
            ));
            return ['error' => $ex->getMessage()];
        }

    }

    function emailRedirectsForService($params)
    {
        $this->domainId = $params['domainId'];
        $this->userId   = $this->getUserIdByDomainId($params['domainId']);
        $this->domainPolicy();

        $domain           = $this->domain->toArray();
        $domain['domain'] = $domain['Domain'];
        unset($domain['Domain']);

        $submoduleId = $this->domain()->domainPricing()->domainSetting()->getSubmodule();
        $submodule   = new main\models\SubmoduleConfiguration($submoduleId);
        if (!$submodule->isActive())
        {
            throw new main\mgLibs\exceptions\System("No access to that domain.");
        }
        try{
            return $this->submodule()->listEmailSingleRedirects($domain);
        }
        catch (\Exception $ex)
        {
            Log::error(sprintf("API: Email Forwarder list failed '%s' - Domain ID: %s - IP: %s\r\n",
                $ex->getMessage(),
                main\mgLibs\helpers\Logger::getDomainUrl($this->domain()->getUserId(), $this->domain()->getId()),
                $this->getRealIpAddr()
            ));
            return [];
        }

    }

    function deleteEmailRedirectsForService($params)
    {
        $required   = [
            'dest',
            'html_forward'
        ];
        $dataNeeded = $this->paramsNeededValidate($_POST, $required);
        if (!empty($dataNeeded))
        {
            return ['error' => $dataNeeded];
        }
        $input['forward'] = [
            'dest'      => $_POST['dest'],
            'sourceurl' => $_POST['html_forward'],
        ];

        $this->domainId = $params['domainId'];
        $this->userId   = $this->getUserIdByDomainId($params['domainId']);
        $this->domainPolicy();
        $params = [
            "domain"  => $this->domain()->toArray(false),
            "forward" => $input['forward'],

        ];
        try
        {
            $msg = $this->submodule()->deleteEmailSingleRedirect($params);

            if (is_string($msg))
            {
                throw new \Exception($msg);
            }
            Log::success(sprintf("API: Email Forwarder has been deleted - Domain ID: %s - IP: %s\r\n",
                main\mgLibs\helpers\Logger::getDomainUrl($this->domain()->getDomain(), $this->domain()->getId()),
                $this->getRealIpAddr()
            ));
            return ['success' => 'Deleted'];
        }
        catch (\Exception $ex)
        {
            Log::error(sprintf("API: Email Forwarder deleting failed '%s' - Domain ID: %s - IP: %s\r\n",
                $ex->getMessage(),
                main\mgLibs\helpers\Logger::getDomainUrl($this->domain()->getUserId(), $this->domain()->getId()),
                $this->getRealIpAddr()
            ));
            return ['error' => $ex->getMessage()];
        }

    }

    function createEmailRedirectsForService($params)
    {
        $required   = [
            'html_forward',
            'dest'
        ];
        $dataNeeded = $this->paramsNeededValidate($_POST, $required);
        if (!empty($dataNeeded))
        {
            return ['error' => $dataNeeded];
        }
        $this->domainId = $params['domainId'];
        $this->userId   = $this->getUserIdByDomainId($params['domainId']);
        $this->domainPolicy();
        $domain_name = \MGModule\DomainAndEmailForwarding\mgLibs\MySQL\Query::select(['domain'], 'tbldomains', ['id' => $this->domainId])
            ->fetchColumn();
        $data        = [
            'source_email' => $_POST['html_forward'],
            'dest_email'   => $_POST['dest'],
            'domain'       => $domain_name,
        ];
        unset($params);
        $params = [
            "domain"  => $this->domain()->toArray(false),
            'forward' => $data,
        ];
        try
        {
            $msg = $this->submodule()->createEmailSingleRedirect($params);

            if (is_string($msg))
            {
                throw new \Exception($msg);
            }
            Log::success(sprintf("API: Email Forwarder has been created - Domain ID: %s - IP: %s\r\n",
                main\mgLibs\helpers\Logger::getDomainUrl($this->domain()->getUserId(), $this->domain()->getId()),
                $this->getRealIpAddr()
            ));
            return ['success' => 'Email redirect created!'];
        }
        catch (\Exception $ex)
        {
            Log::error(sprintf("API: Email Forwarder creating failed '%s' - Domain ID: %s - IP: %s\r\n",
                $ex->getMessage(),
                main\mgLibs\helpers\Logger::getDomainUrl($this->domain()->getUserId(), $this->domain()->getId()),
                $this->getRealIpAddr()
            ));
            return ['error' => $ex->getMessage()];
        }

    }

    function domainRedirectsForService($params)
    {

        $this->domainId = $params['domainId'];
        $this->userId   = $this->getUserIdByDomainId($params['domainId']);
        $this->domainPolicy();

        $params = ["domain" => $this->domain()->toArray(false)];
        try
        {
            return $this->submodule()->listDomainRedirects($params);
        }
        catch (\Exception $ex)
        {
            Log::error(sprintf("API: Domain Redirect list failed '%s' - Domain ID: %s - IP: %s\r\n",
                $ex->getMessage(),
                main\mgLibs\helpers\Logger::getDomainUrl($this->domain()->getUserId(), $this->domain()->getId()),
                $this->getRealIpAddr()
            ));
            return ['error' => $ex->getMessage()];
        }

    }

    public function paramsNeededValidate($data, $dataNeeded)
    {

        foreach ($data as $key => $value)
        {
            if (in_array($key, $dataNeeded))
            {
                unset($dataNeeded[array_search($key, $dataNeeded)]);
            }
        }
        $out = [];
        foreach ($dataNeeded as $key => $value)
        {
            $out[$value] = 'is required';
        }
        return $out;
    }


    public function createDomainRedirectsForService($params, $vars = [])
    {

        $required   = [
            'type',
            'url',
            'src'
        ];
        $dataNeeded = $this->paramsNeededValidate($_POST, $required);
        if (!empty($dataNeeded))
        {
            return ['error' => $dataNeeded];
        }
        $this->domainId = $params['domainId'];
        $this->userId   = $this->getUserIdByDomainId($params['domainId']);
        $this->domainPolicy();
        $input['forward'] = [
            'type'     => $_POST['type'],
            'domain'   => \MGModule\DomainAndEmailForwarding\mgLibs\MySQL\Query::select(['domain'], 'tbldomains', ['id' => $params['domainId']])->fetchColumn(),
            'url'      => $_POST['url'],
            'rdwww'    => $_POST['rdww'] ? (int) $_POST['rdww'] : '',
            'wildcard' => $_POST['wildcard'] ? (int) $_POST['wildcard'] : '',
            'src'      => $_POST['src'],
        ];

        $createParams                 = [
            "domain"  => $this->domain()->toArray(false),
            'forward' => $input['forward']
        ];
        try
        {
            $msg = $this->submodule()->createDomainRedirect($createParams);
            if (is_string($msg))
            {
                throw new \Exception($msg);
            }
            Log::success(sprintf("API: Domain Redirect has been created - Domain ID: %s - IP: %s\r\n",
                main\mgLibs\helpers\Logger::getDomainUrl($this->domain()->getUserId(), $this->domain()->getId()),
                $this->getRealIpAddr()
                ));
            return ["success" => main\mgLibs\Lang::absoluteT('forwarddomains', 'add_dforward_info_text')];
        }
        catch (\Exception $ex)
        {
            Log::error(sprintf("API: Domain Redirect createing failed '%s' - Domain ID: %s - IP: %s\r\n",
                $ex->getMessage(),
                main\mgLibs\helpers\Logger::getUserUrl($this->domain()->getUserId()),
                main\mgLibs\helpers\Logger::getDomainUrl($this->domain()->getUserId(), $this->domain()->getId()),
                $this->getRealIpAddr()
            ));
            return ['error' => $ex->getMessage()];
        }
    }

    public function deleteDomainRedirectsForService($params, $vars = [])
    {
        $required   = ['sourceurl'];
        $dataNeeded = $this->paramsNeededValidate($_POST, $required);
        if (!empty($dataNeeded))
        {
            return ['error' => $dataNeeded];
        }
        $this->domainId   = ($params['domainId']);
        $this->userId     = $this->getUserIdByDomainId($params['domainId']);
        $sourceURL        = ($_POST['sourceurl'] == '/') ? '' : $_POST['sourceurl'];
        $input['forward'] = [
            'sourceurl' => '/' . $sourceURL,
            'docroot'   => '/home/' . $this->getResseler() . '/public_html',
        ];

        $this->domainPolicy();
        $paramsToDelete = [
            "domain"  => $this->domain()->toArray(false),
            "forward" => $input['forward']
        ];
        try
        {
            $msg = $this->submodule()->deleteDomainRedirect($paramsToDelete);
            if (is_string($msg))
            {
                throw new \Exception($msg);
            }
            Log::success(sprintf("API: Domain Redirect has been deleted - Domain ID: %s - IP: %s\r\n",
                main\mgLibs\helpers\Logger::getDomainUrl($this->domain()->getUserId(), $this->domain()->getId()),
                $this->getRealIpAddr()
            ));

            return ['success' => 'Domain has been deleted!'];
        }
        catch (\Exception $ex)
        {
            Log::error(sprintf("API: Domain Redirect deleting failed '%s' - Domain ID: %s - IP: %s\r\n",
                $ex->getMessage(),
                main\mgLibs\helpers\Logger::getDomainUrl($this->domain()->getUserId(), $this->domain()->getId()),
                $this->getRealIpAddr()
            ));
            return ['error' => $ex->getMessage()];
        }

    }

    public function getUserIdByDomainId($domainId)
    {
        return \MGModule\DomainAndEmailForwarding\mgLibs\MySQL\Query::select(['userId'], 'tbldomains', ['id' => $domainId])
            ->fetchColumn('userId');
    }

    public function getResseler()
    {
        $config = $this->getSubmoduleConfiguration();
        return $config['reseller'];
    }

    public function getSubmoduleConfiguration()
    {
        $submodule = $this->submodule();
        $array     = (array)$submodule;
        foreach ($array as $a)
        {
            return $a;
        }
    }

    private function getRealIpAddr()
    {
        if (!empty($_SERVER['HTTP_CLIENT_IP']))   //check ip from share internet
        {
            return $_SERVER['HTTP_CLIENT_IP'];
        }
        else if (!empty($_SERVER['HTTP_X_FORWARDED_FOR']))   //to check ip is pass from proxy
        {
            return $_SERVER['HTTP_X_FORWARDED_FOR'];
        }
        return $_SERVER['REMOTE_ADDR'];
    }


}
